const express = require('express');
const axios = require('axios');
const cors = require('cors');

const app = express();
const port = 3001;

app.use(cors({
  origin: '*',
  methods: ['GET']
}));

// Route to fetch METAR from NOAA
app.get('/api/metar/:icao', async (req, res) => {
  const icao = req.params.icao.toUpperCase();
  const url = `https://tgftp.nws.noaa.gov/data/observations/metar/stations/${icao}.TXT`;

  console.log(`[METAR] Received request for ${icao}`);
  console.log(`[METAR] Fetching: ${url}`);

  try {
    const response = await axios.get(url);
    res.type('text/plain').send(response.data);
  } catch (error) {
    console.error(`[METAR] Error for ${icao}:`, error.message);
    res.status(500).send(`METAR not available: ${error.message}`);
  }
});

// Route to fetch TAF from NOAA
app.get('/api/taf/:icao', async (req, res) => {
  const icao = req.params.icao.toUpperCase();
  const url = `https://tgftp.nws.noaa.gov/data/forecasts/taf/stations/${icao}.TXT`;

  try {
    const response = await axios.get(url);
    res.type('text/plain').send(response.data);
  } catch (error) {
    console.error(`Failed to fetch TAF for ${icao}:`, error.message);
    res.status(404).send(`TAF not available for ${icao}`);
  }
});

// Route to fetch local forecast from WeatherAPI
app.get('/api/forecast', async (req, res) => {
  const location = req.query.location || 'Carrigaline';
  const apiKey = '11305e465cb7402c8b5223240250504';
  const url = `http://api.weatherapi.com/v1/forecast.json?key=${apiKey}&q=${location}&days=5&aqi=no&alerts=no`;

  try {
    const response = await axios.get(url);
    res.json(response.data);
  } catch (error) {
    console.error('Error fetching forecast:', error.message);
    res.status(500).json({ error: 'Failed to fetch forecast data' });
  }
});

// Route to fetch live flights from OpenSky (restricted to Cork area)
app.get('/api/flights', async (req, res) => {
  try {
    const response = await axios.get(
      'https://opensky-network.org/api/states/all?lamin=51.7&lomin=-8.6&lamax=52.0&lomax=-8.2',
      {
        auth: {
          username: 'corksafetyalerts',
          password: 'Minnieming18$'
        }
      }
    );
    res.json(response.data);
  } catch (error) {
    console.error('Failed to fetch flight data:', error.message);
    res.status(500).json({ error: 'Flight data unavailable' });
  }
});

// Health check route
const dns = require('dns');
const net = require('net');

app.get('/api/health', async (req, res) => {
  const metarIcao = 'EICK';
  const metarUrl = `https://tgftp.nws.noaa.gov/data/observations/metar/stations/${metarIcao}.TXT`;
  const serverIp = '209.126.156.188';
  const serverPort = 22; // typically SSH port, can change if needed

  let isHealthy = false;
  let metarText = '';
  let isServerReachable = false;

  // METAR check
  try {
    const response = await axios.get(metarUrl);
    metarText = response.data;
    isHealthy = true;
  } catch (err) {
    console.error('Health check METAR fetch failed:', err.message);
  }

  // Server reachability check (TCP ping)
  const checkServerReachability = () =>
    new Promise((resolve) => {
      const socket = new net.Socket();
      socket.setTimeout(3000);
      socket.on('connect', () => {
        socket.destroy();
        resolve(true);
      }).on('error', () => resolve(false))
        .on('timeout', () => resolve(false))
        .connect(serverPort, serverIp);
    });

  isServerReachable = await checkServerReachability();

  const html = `
    <!DOCTYPE html>
    <html lang="en">
    <head>
      <meta charset="UTF-8">
      <title>CSA Weather Server Health</title>
      <style>
        body {
          font-family: Arial, sans-serif;
          background: #f0f4f8;
          color: #333;
          text-align: center;
          padding-top: 60px;
        }
        .status {
          font-size: 2rem;
          padding: 20px;
          border-radius: 12px;
          display: inline-block;
          background: ${isHealthy ? '#d4edda' : '#f8d7da'};
          color: ${isHealthy ? '#155724' : '#721c24'};
          border: 1px solid ${isHealthy ? '#c3e6cb' : '#f5c6cb'};
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
        }
        .subcheck {
          margin-top: 20px;
          font-size: 1rem;
        }
        .metar {
          margin-top: 30px;
          font-size: 0.95rem;
          color: #777;
          white-space: pre-wrap;
        }
      </style>
    </head>
    <body>
      <div class="status">
        ${isHealthy ? '✅ All looks good in Houston, boss!' : '❌ Houston, we have a problem'}
      </div>

      <div class="subcheck">
        ${isServerReachable ? '✅ Server is reachable' : '❌ Server is unreachable'}
      </div>

      ${isHealthy ? `<div class="metar">${metarText.trim()}</div>` : ''}
    </body>
    </html>
  `;

  res.status(isHealthy ? 200 : 500).send(html);
});




// Start the server
app.listen(port, () => {
  console.log(`✅ Weather server running at http://localhost:${port}`);
});
